/* Copyright (C) 2015-2018 RealVNC Ltd. All Rights Reserved.
 */

#ifndef UNIQUE_INTEGER_ID_H_D82CF126_FA39_11E4_B1C2_1B782557EFAD
#define UNIQUE_INTEGER_ID_H_D82CF126_FA39_11E4_B1C2_1B782557EFAD

#include <vncint.h>

#include <sstream>
#include <ostream>

/**
 * \cond VNCCOMMON
 */
namespace vnccommon
{

/**
 * \brief A unique identifier.
 */
template<typename T>
class UniqueIntegerID
{
public:
    /** \brief Less-than operator.
     *
     * \param o The ID to compare against.
     * \return \c true if this ID is lower than \a o.
     */
    inline bool operator<(const UniqueIntegerID& o) const
    {
        return mValue < o.mValue;
    }

    /** \brief Greater-than operator.
     *
     * \param o The ID to compare against.
     * \return \c true if this ID is higher than \a o.
     */
    inline bool operator>(const UniqueIntegerID& o) const
    {
        return mValue > o.mValue;
    }

    /** \brief Less-than-or-equal operator.
     *
     * \param o The ID to compare against.
     * \return \c true if this ID is lower than or equal to \a o.
     */
    inline bool operator<=(const UniqueIntegerID& o) const
    {
        return mValue <= o.mValue;
    }

    /** \brief Greater-than-or-equal operator.
     *
     * \param o The ID to compare against.
     * \return \c true if this ID is higher than or equal to \a o.
     */
    inline bool operator>=(const UniqueIntegerID& o) const
    {
        return mValue >= o.mValue;
    }

    /** \brief Equal operator.
     *
     * \param o The ID to compare against.
     * \return \c true if this ID is the same as \a o.
     */
    inline bool operator==(const UniqueIntegerID& o) const
    {
        return mValue == o.mValue;
    }

    /** \brief Not-equal operator.
     *
     * \param o The ID to compare against.
     * \return \c true if this ID is not the same as \a o.
     */
    inline bool operator!=(const UniqueIntegerID& o) const
    {
        return mValue != o.mValue;
    }

    /**
     * \brief Returns the integer value of the ID.
     *
     * \return The integer value.
     */
    vnc_uint64_t toInteger() const
    {
        return mValue;
    }

    /**
     * \brief Returns a string representation of the ID.
     *
     * \return A string representation.
     */
    std::string toString() const
    {
        std::ostringstream result;
        result << mValue;
        return result.str();
    }

    /**
     * \brief Constructs an ID from an integer value.
     *
     * \param value The integer value.
     */
    explicit UniqueIntegerID(vnc_uint64_t value)
        : mValue(value)
    {
    }

    /**
     * \brief Default constructor, for use with \ref Optional.
     */
    UniqueIntegerID()
        : mValue(0)
    {
    }

private:
    vnc_uint64_t mValue;
};

} // end of namespace vnccommon
/**
 * \endcond
 */

#endif /* !define(UNIQUE_INTEGER_ID_H_D82CF126_FA39_11E4_B1C2_1B782557EFAD) */
